/** @file
  Set up ROM Table for PCI Bus module.

Copyright (c) 2006 - 2017, Intel Corporation. All rights reserved.<BR>
This program and the accompanying materials
are licensed and made available under the terms and conditions of the BSD License
which accompanies this distribution.  The full text of the license may be found at
http://opensource.org/licenses/bsd-license.php

THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

#include "PciBus.h"

//
// PCI ROM image information
//
typedef struct {
  EFI_HANDLE  ImageHandle;
  UINTN       Seg;
  UINT8       Bus;
  UINT8       Dev;
  UINT8       Func;
  VOID        *RomImage; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  UINT64      RomSize;   // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
} PCI_ROM_IMAGE;         // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

UINTN          mNumberOfPciRomImages     = 0;    // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
UINTN          mMaxNumberOfPciRomImages  = 0;    // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
PCI_ROM_IMAGE  *mRomImageTable           = NULL; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

/**
  Add the Rom Image to internal database for later PCI light enumeration.

  @param ImageHandle    Option Rom image handle.
  @param Seg            Segment of PCI space.
  @param Bus            Bus NO of PCI space.
  @param Dev            Dev NO of PCI space.
  @param Func           Func NO of PCI space.
  @param RomImage       Option Rom buffer.         // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  @param RomSize        Size of Option Rom buffer. // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
**/
VOID
PciRomAddImageMapping (
  IN  EFI_HANDLE  ImageHandle,
  IN  UINTN       Seg,
  IN  UINT8       Bus,
  IN  UINT8       Dev,
  IN  UINT8       Func,
  IN  VOID        *RomImage, // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  IN  UINT64      RomSize    // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
  )
{
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideBegin - RPPO-KBL-0087
//
//  EFI_PCI_ROM_IMAGE_MAPPING *TempMapping;
//
//  if (mNumberOfPciRomImages >= mMaxNumberOfPciRomImages) {
//
//    mMaxNumberOfPciRomImages += 0x20;
//
//    TempMapping = NULL;
//    TempMapping = AllocatePool (mMaxNumberOfPciRomImages * sizeof (EFI_PCI_ROM_IMAGE_MAPPING));
//    if (TempMapping == NULL) {
//      return ;
//    }
//
//    CopyMem (TempMapping, mRomImageTable, mNumberOfPciRomImages * sizeof (EFI_PCI_ROM_IMAGE_MAPPING));
//
//    if (mRomImageTable != NULL) {
//      FreePool (mRomImageTable);
//    }
//
//    mRomImageTable = TempMapping;
//  }

//  mRomImageTable[mNumberOfPciRomImages].ImageHandle = ImageHandle;
//  mRomImageTable[mNumberOfPciRomImages].Seg         = Seg;
//  mRomImageTable[mNumberOfPciRomImages].Bus         = Bus;
//  mRomImageTable[mNumberOfPciRomImages].Dev         = Dev;
//  mRomImageTable[mNumberOfPciRomImages].Func        = Func;
//  mRomImageTable[mNumberOfPciRomImages].RomAddress  = RomAddress;
//  mRomImageTable[mNumberOfPciRomImages].RomLength   = RomLength;
//  mNumberOfPciRomImages++;
  UINTN           Index;
  PCI_ROM_IMAGE   *NewTable;

  for (Index = 0; Index < mNumberOfPciRomImages; Index++) {
    if (mRomImageTable[Index].Seg  == Seg &&
        mRomImageTable[Index].Bus  == Bus &&
        mRomImageTable[Index].Dev  == Dev &&
        mRomImageTable[Index].Func == Func) {
      //
      // Expect once RomImage and RomSize are recorded, they will be passed in
      // later when updating ImageHandle
      //
      ASSERT ((mRomImageTable[Index].RomImage == NULL) || (RomImage == mRomImageTable[Index].RomImage));
      ASSERT ((mRomImageTable[Index].RomSize  == 0   ) || (RomSize  == mRomImageTable[Index].RomSize ));
      break;
    }
  }

  if (Index == mNumberOfPciRomImages) {
    //
    // Rom Image Table buffer needs to grow.
    //
    if (mNumberOfPciRomImages == mMaxNumberOfPciRomImages) {
      NewTable = ReallocatePool (
                   mMaxNumberOfPciRomImages * sizeof (PCI_ROM_IMAGE),
                   (mMaxNumberOfPciRomImages + 0x20) * sizeof (PCI_ROM_IMAGE),
                   mRomImageTable
                   );
      if (NewTable == NULL) {
        return ;
      }

      mRomImageTable            = NewTable;
      mMaxNumberOfPciRomImages += 0x20;
    }
    //
    // Record the new PCI device
    //
    mRomImageTable[Index].Seg  = Seg;
    mRomImageTable[Index].Bus  = Bus;
    mRomImageTable[Index].Dev  = Dev;
    mRomImageTable[Index].Func = Func;
    mNumberOfPciRomImages++;
  }

  mRomImageTable[Index].ImageHandle = ImageHandle;
  mRomImageTable[Index].RomImage    = RomImage;
  mRomImageTable[Index].RomSize     = RomSize;
//
// ROYAL_PARK_OVERRIDE: RoyalParkOverrideEnd - RPPO-KBL-0087
//
}

/**
  Get Option rom driver's mapping for PCI device.

  @param PciIoDevice Device instance.

  @retval TRUE   Found Image mapping.
  @retval FALSE  Cannot found image mapping.

**/
BOOLEAN
PciRomGetImageMapping (
  IN  PCI_IO_DEVICE                       *PciIoDevice
  )
{
  EFI_PCI_ROOT_BRIDGE_IO_PROTOCOL *PciRootBridgeIo;
  UINTN                           Index;
  //BOOLEAN                         Found; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  PciRootBridgeIo = PciIoDevice->PciRootBridgeIo;
  //Found           = FALSE; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

  for (Index = 0; Index < mNumberOfPciRomImages; Index++) {
    if (mRomImageTable[Index].Seg  == PciRootBridgeIo->SegmentNumber &&
        mRomImageTable[Index].Bus  == PciIoDevice->BusNumber         &&
        mRomImageTable[Index].Dev  == PciIoDevice->DeviceNumber      &&
        mRomImageTable[Index].Func == PciIoDevice->FunctionNumber    ) {
        //Found = TRUE; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087

      if (mRomImageTable[Index].ImageHandle != NULL) {
        AddDriver (PciIoDevice, mRomImageTable[Index].ImageHandle, NULL);                  // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      //} else {                                                                           // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      //  PciIoDevice->PciIo.RomImage = (VOID *) (UINTN) mRomImageTable[Index].RomAddress; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      //  PciIoDevice->PciIo.RomSize  = (UINTN) mRomImageTable[Index].RomLength;           // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      }
      PciIoDevice->PciIo.RomImage = mRomImageTable[Index].RomImage; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      PciIoDevice->PciIo.RomSize  = mRomImageTable[Index].RomSize;  // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
      return TRUE;                                                  // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
    }
  }

  return FALSE; // ROYAL_PARK_OVERRIDE: RoyalParkOverrideContent - RPPO-KBL-0087
}
